<?php
/**
 * Update Contact
 *
 * @package     AutomatorWP\Integrations\HubSpot\Actions\Update_Contact
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_HubSpot_Update_Contact extends AutomatorWP_Integration_Action {

    public $integration = 'hubspot';
    public $action = 'hubspot_update_contact';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Update a contact', 'automatorwp-pro' ),
            'select_option'     => __( 'Update a <strong>contact</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Contact. */
            'edit_label'        => sprintf( __( 'Update %1$s', 'automatorwp-pro' ), '{contact}' ),
            /* translators: %1$s: Contact. */
            'log_label'         => sprintf( __( 'Update %1$s', 'automatorwp-pro' ), '{contact}' ),
            'options'           => array(
                'contact' => array(
                    'from' => 'email',
                    'default' => __( 'contact', 'automatorwp-pro' ),
                    'fields' => array(
                        'email' => array(
                            'name' => __( 'Email:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required' => true,
                        ),
                        'first_name' => array(
                            'name' => __( 'First Name:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'last_name' => array(
                            'name' => __( 'Last Name:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'phone' => array(
                            'name' => __( 'Phone:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'website' => array(
                            'name' => __( 'Website:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'company' => array(
                            'name' => __( 'Company:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'address' => array(
                            'name' => __( 'Address:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'city' => array(
                            'name' => __( 'City:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'zip' => array(
                            'name' => __( 'Zip Code:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $fields = array(
            'first_name' => '',
            'last_name' => '',
            'email' => '',
            'phone' => '',
            'website' => '',
            'company' => '',
            'address' => '',
            'city' => '',
            'zip' => ''
        );

        $contact_data = wp_parse_args( $action_options, $fields );

        $params = automatorwp_hubspot_get_request_parameters();

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            $this->result = __( 'HubSpot authorization not setup yet.', 'automatorwp-pro' );
            return;
        }

        // Bail if email is not correctly setup
        if ( ! is_email( $contact_data['email'] ) ) {
            $this->result = sprintf( __( 'The email %s is not a valid email.', 'automatorwp-pro' ), $contact_data['email'] );
            return;
        }

        // Setup the contact properties
        $properties = array();

        // Field mapping
        $field_maps = array(
            'first_name' => 'firstname',
            'last_name' => 'lastname',
        );

        foreach( $contact_data as $field => $value ) {

            // Skip not contact fields
            if( ! isset( $fields[$field] ) ) {
                continue;
            }

            // Map field names
            if( isset( $field_maps[$field] ) ) {
                $field = $field_maps[$field];
            }

            $properties[] = array(
                'property' => $field,
                'value' => $value,
            );
        }

        $params['body'] = json_encode( array( 'properties' => $properties ) );
        
        $url = 'https://api.hubapi.com/contacts/v1/contact/email/' . $contact_data['email'] . '/profile';

        // Execute the request
        $response = wp_remote_post( $url, $params );
    
        // Bail on error
        if( is_wp_error( $response ) ) {
            $this->result = sprintf( __( 'Could not update the contact, error: %s.', 'automatorwp-pro' ), $response->get_error_message() );
            return;
        }

        $response_code = wp_remote_retrieve_response_code( $response );

        // Setup result based on response code
        if( $response_code === 204 ) {
            $this->result = __( 'Contact updated on HubSpot successfully.', 'automatorwp-pro' );
        } else if( $response_code === 404 ) {
            $this->result = sprintf( __( 'Could not update the contact because the contact with the email %s does not exist.', 'automatorwp-pro' ), $contact_data['email'] );
        } else if( isset( $response_body['message'] ) ) {
            $this->result = sprintf( __( 'HubSpot responded with the error: %s.', 'automatorwp-pro' ), $response_body['message'] );
        } else {

            $response_body = json_decode( wp_remote_retrieve_body( $response ), true );

            if( isset( $response_body['message'] ) ) {
                $this->result = sprintf( __( 'HubSpot responded with the error: %s.', 'automatorwp-pro' ), $response_body['message'] );
            } else {
                $this->result = sprintf( __( 'HubSpot responded with the error code: %d.', 'automatorwp-pro' ), $response_code );
            }


        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $contact_data = wp_parse_args( $action_options, array(
            'first_name' => '',
            'last_name' => '',
            'email' => '',
            'phone' => '',
            'website' => '',
            'company' => '',
            'address' => '',
            'city' => '',
            'zip' => ''
        ) );

        // Store the contact data
        foreach( $contact_data as $field => $value ) {
            $log_meta[$field] = $value;
        }

        // Store the action's result
        $log_meta['result']     = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['contact_data'] = array(
            'name' => __( 'Contact Data', 'automatorwp-pro' ),
            'desc' => __( 'Information about contact information sent.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['first_name'] = array(
            'name' => __( 'First Name:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['last_name'] = array(
            'name' => __( 'Last Name:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['email'] = array(
            'name' => __( 'Email:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['phone'] = array(
            'name' => __( 'Phone:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['website'] = array(
            'name' => __( 'Website:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['company'] = array(
            'name' => __( 'Company:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['address'] = array(
            'name' => __( 'Address:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['city'] = array(
            'name' => __( 'City:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['zip'] = array(
            'name' => __( 'Zip Code:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_HubSpot_Update_Contact();